/*
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch and DENSO.
 *
 * This file is part of COVESA Project DLT - Diagnostic Log and Trace.
 *
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License (MPL), v. 2.0.
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * For further information see https://covesa.global.
 */

/*!
 * \author
 * Le Van Khanh <Khanh.LeVan@vn.bosch.com>
 *
 * \copyright Copyright © 2023 Advanced Driver Information Technology. \n
 * License MPL-2.0: Mozilla Public License version 2.0 http://mozilla.org/MPL/2.0/.
 *
 * \file gtest_dlt_daemon_statistics.cpp
 */

/*******************************************************************************
**                                                                            **
**  SRC-MODULE: gtest_dlt_daemon_statistics.cpp                               **
**                                                                            **
**  TARGET    : linux                                                         **
**                                                                            **
**  PROJECT   : DLT                                                           **
**                                                                            **
**  AUTHOR    : Le Van Khanh <Khanh.LeVan@vn.bosch.com>                       **
**                                                                            **
**  PURPOSE   :                                                               **
**                                                                            **
**  REMARKS   :                                                               **
**                                                                            **
**  PLATFORM DEPENDANT [yes/no]: yes                                          **
**                                                                            **
**  TO BE CHANGED BY USER [yes/no]: no                                        **
**                                                                            **
*******************************************************************************/

/*******************************************************************************
**                      Author Identity                                       **
********************************************************************************
**                                                                            **
** Initials     Name                       Company                            **
** --------     -------------------------  ---------------------------------- **
**  cl          Le Van Khanh               Bosch                              **
*******************************************************************************/

#include <gtest/gtest.h>
#include <regex>
#include <fstream>
#include <stdlib.h>
#if defined(DLT_LOG_STATISTIC) && defined(BOSCH_ECP_543_RMAP_2452492_LOG_ENABLE_SPAM_FILTERING)
extern "C"
{
    #include "dlt_daemon_statistic.h"
    #include "dlt_daemon_statistic.c"
    #include "string.h"
    #include <search.h>
    #include <stdarg.h>
    #include "dlt_gateway.h"
    #include "dlt_gateway_internal.h"
    #include "dlt_daemon_client.h"
    int mock_dlt_log_return = DLT_RETURN_OK;
    int mock_dlt_daemon_log_internal_statistics_return = 0;
    int mock_dlt_vlog_called = 0;
    int mock_dlt_log_called = 0;
    int mock_dlt_daemon_log_internal_statistics_called = 0;
    DltReturnValue dlt_log(int level, char *str) {
        mock_dlt_log_called++;
        (void)level; (void)str;
        return (DltReturnValue)mock_dlt_log_return;
    }
    DltReturnValue dlt_vlog(int level, const char *fmt, ...) {
        mock_dlt_vlog_called++;
        (void)level; (void)fmt;
        return (DltReturnValue)0;
    }
    int dlt_daemon_log_internal_statistics(DltDaemon *daemon, DltDaemonLocal *daemon_local, char *str, int vflag) {
        mock_dlt_daemon_log_internal_statistics_called++;
        (void)daemon; (void)daemon_local; (void)str; (void)vflag;
        return mock_dlt_daemon_log_internal_statistics_return;
    }
}

// Unit tests for dlt_daemon_statistic_print_to_log_specific
TEST(t_dlt_daemon_statistic_print_to_log_specific, null_string)
{
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_to_log_specific(NULL);
    EXPECT_EQ(ret, -1);
    EXPECT_GT(mock_dlt_vlog_called, 0);
}

TEST(t_dlt_daemon_statistic_print_to_log_specific, log_failed_no_internal)
{
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    // Set internals to NULL
    extern DltDaemon *daemon_internal;
    extern DltDaemonLocal *daemon_local_internal;
    daemon_internal = NULL;
    daemon_local_internal = NULL;
    int ret = dlt_daemon_statistic_print_to_log_specific((char*)"Test log");
    EXPECT_EQ(ret, -1);
    EXPECT_EQ(mock_dlt_log_called, 1);
}

TEST(t_dlt_daemon_statistic_print_to_log_specific, log_fail)
{
    mock_dlt_log_return = -1;
    mock_dlt_log_called = 0;
    int ret = dlt_daemon_statistic_print_to_log_specific((char*)"Test log");
    EXPECT_EQ(ret, -1);
    EXPECT_EQ(mock_dlt_log_called, 1);
}

TEST(t_dlt_daemon_statistic_print_to_log_specific, internal_statistics_fail)
{
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_daemon_log_internal_statistics_return = 1;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    // Set internals to non-NULL
    static DltDaemon dummy_daemon;
    static DltDaemonLocal dummy_local;
    extern DltDaemon *daemon_internal;
    extern DltDaemonLocal *daemon_local_internal;
    daemon_internal = &dummy_daemon;
    daemon_local_internal = &dummy_local;
    int ret = dlt_daemon_statistic_print_to_log_specific((char*)"Test log");
    EXPECT_EQ(ret, -1);
    EXPECT_EQ(mock_dlt_daemon_log_internal_statistics_called, 1);
}

TEST(t_dlt_daemon_statistic_print_to_log_specific, internal_statistics_success)
{
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    static DltDaemon dummy_daemon;
    static DltDaemonLocal dummy_local;
    extern DltDaemon *daemon_internal;
    extern DltDaemonLocal *daemon_local_internal;
    daemon_internal = &dummy_daemon;
    daemon_local_internal = &dummy_local;
    int ret = dlt_daemon_statistic_print_to_log_specific((char*)"Test log");
    EXPECT_EQ(ret, 0);
    EXPECT_EQ(mock_dlt_daemon_log_internal_statistics_called, 1);
}

/* Unit test for dlt_daemon_statistic_format_time_line (normal path) */
TEST(t_dlt_daemon_statistic_format_time_line, normal_path)
{
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;

    // Should succeed and return 0
    int ret = dlt_daemon_statistic_format_time_line();
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

TEST(t_dlt_daemon_statistic_format_time_line, buffer_overflow_time_header)
{
    extern char *time_print_element[];
    char long_str[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH * 2];
    memset(long_str, 'A', sizeof(long_str) - 1);
    long_str[sizeof(long_str) - 1] = '\0';
    char *orig = time_print_element[0];
    time_print_element[0] = long_str;
    int ret = dlt_daemon_statistic_format_time_line();
    EXPECT_EQ(ret, -1);
    time_print_element[0] = orig;
}

// Error-path: Simulate negative spaces calculation
TEST(t_dlt_daemon_statistic_format_time_line, negative_spaces_needed)
{
    extern int tablength;
    int orig_tablength = tablength;
    tablength = 0;
    int ret = dlt_daemon_statistic_format_time_line();
    EXPECT_EQ(ret, -1);
    tablength = orig_tablength;
}

TEST(t_dlt_daemon_statistic_format_time_line, all_time_print_element_empty)
{
    extern char *time_print_element[];
    char *orig[4];
    char empty[] = "";
    for (int i = 0; i < 4; ++i) {
        orig[i] = time_print_element[i];
        time_print_element[i] = empty;
    }
    int ret = dlt_daemon_statistic_format_time_line();
    EXPECT_EQ(ret, 0);
    for (int i = 0; i < 4; ++i) {
        time_print_element[i] = orig[i];
    }
}

// Unit tests for dlt_daemon_statistic_print_monotonic_time
TEST(t_dlt_daemon_statistic_print_monotonic_time, normal_path)
{
    struct timespec m_start = { .tv_sec = 100, .tv_nsec = 5000000 };
    struct timespec m_end = { .tv_sec = 105, .tv_nsec = 8000000 };
    uint32_t duration = 5;
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_monotonic_time(m_start, m_end, duration);
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

// Error-path: Simulate negative spaces calculation
TEST(t_dlt_daemon_statistic_print_monotonic_time, negative_spaces_needed)
{
    extern int tablength;
    int orig_tablength = tablength;
    tablength = 0;
    struct timespec m_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec m_end = { .tv_sec = 2, .tv_nsec = 2 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_monotonic_time(m_start, m_end, duration);
    EXPECT_EQ(ret, -1);
    tablength = orig_tablength;
}

// Edge-case: all time_print_element strings empty
TEST(t_dlt_daemon_statistic_print_monotonic_time, all_time_print_element_empty)
{
    extern char *time_print_element[];
    char *orig[4];
    char empty[] = "";
    for (int i = 0; i < 4; ++i) {
        orig[i] = time_print_element[i];
        time_print_element[i] = empty;
    }
    struct timespec m_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec m_end = { .tv_sec = 2, .tv_nsec = 2 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_monotonic_time(m_start, m_end, duration);
    EXPECT_EQ(ret, 0);
    for (int i = 0; i < 4; ++i) {
        time_print_element[i] = orig[i];
    }
}

// Unit tests for dlt_daemon_statistic_print_system_time
TEST(t_dlt_daemon_statistic_print_system_time, normal_path)
{
    struct timespec s_start = { .tv_sec = 200, .tv_nsec = 1000000 };
    struct timespec s_end = { .tv_sec = 205, .tv_nsec = 2000000 };
    uint32_t duration = 5;
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_system_time(s_start, s_end, duration);
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

// Error-path: Simulate negative spaces calculation
TEST(t_dlt_daemon_statistic_print_system_time, negative_spaces_needed)
{
    extern int tablength;
    int orig_tablength = tablength;
    tablength = 0;
    struct timespec s_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec s_end = { .tv_sec = 2, .tv_nsec = 2 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_system_time(s_start, s_end, duration);
    EXPECT_EQ(ret, -1);
    tablength = orig_tablength;
}

// Test with zero duration
TEST(t_dlt_daemon_statistic_print_system_time, zero_duration)
{
    struct timespec s_start = { .tv_sec = 100, .tv_nsec = 5000000 };
    struct timespec s_end = { .tv_sec = 100, .tv_nsec = 5000000 };
    uint32_t duration = 0;
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_system_time(s_start, s_end, duration);
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

// Test with maximum valid 5-digit timestamp values (format uses %5u)
TEST(t_dlt_daemon_statistic_print_system_time, large_timestamps)
{
    struct timespec s_start = { .tv_sec = 99999, .tv_nsec = 999999999 };
    struct timespec s_end = { .tv_sec = 99999, .tv_nsec = 999999999 };
    uint32_t duration = 100;
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_system_time(s_start, s_end, duration);
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

// Test failure in dlt_log
TEST(t_dlt_daemon_statistic_print_system_time, dlt_log_failure)
{
    struct timespec s_start = { .tv_sec = 100, .tv_nsec = 5000000 };
    struct timespec s_end = { .tv_sec = 105, .tv_nsec = 8000000 };
    uint32_t duration = 5;
    // Set mock to fail
    mock_dlt_log_return = DLT_RETURN_ERROR;
    mock_dlt_log_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_system_time(s_start, s_end, duration);
    EXPECT_EQ(ret, -1);
    EXPECT_GE(mock_dlt_log_called, 1);
    // Reset mock
    mock_dlt_log_return = DLT_RETURN_OK;
}

// Unit tests for dlt_daemon_statistic_print_time_info
TEST(t_dlt_daemon_statistic_print_time_info, normal_path)
{
    struct timespec m_start = { .tv_sec = 100, .tv_nsec = 5000000 };
    struct timespec m_end = { .tv_sec = 105, .tv_nsec = 8000000 };
    struct timespec s_start = { .tv_sec = 200, .tv_nsec = 1000000 };
    struct timespec s_end = { .tv_sec = 205, .tv_nsec = 2000000 };
    uint32_t duration = 5;
    // Reset mock counters
    mock_dlt_log_return = DLT_RETURN_OK;
    mock_dlt_log_called = 0;
    mock_dlt_daemon_log_internal_statistics_return = 0;
    mock_dlt_daemon_log_internal_statistics_called = 0;
    mock_dlt_vlog_called = 0;
    int ret = dlt_daemon_statistic_print_time_info(m_start, m_end, s_start, s_end, duration);
    EXPECT_EQ(ret, 0);
    EXPECT_GE(mock_dlt_log_called, 1);
}

// Error-path: Simulate tablength = 0 (should fail)
TEST(t_dlt_daemon_statistic_print_time_info, invalid_tablength)
{
    extern int tablength;
    int orig_tablength = tablength;
    tablength = 0;
    struct timespec m_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec m_end = { .tv_sec = 2, .tv_nsec = 2 };
    struct timespec s_start = { .tv_sec = 3, .tv_nsec = 3 };
    struct timespec s_end = { .tv_sec = 4, .tv_nsec = 4 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_time_info(m_start, m_end, s_start, s_end, duration);
    EXPECT_EQ(ret, -1);
    tablength = orig_tablength;
}

// Error-path: Simulate failure in dlt_daemon_statistic_format_time_line
TEST(t_dlt_daemon_statistic_print_time_info, format_time_line_failure)
{
    // Patch tablength to 0 to force dlt_daemon_statistic_format_time_line to fail
    extern int tablength;
    int orig_tablength = tablength;
    tablength = 0;
    struct timespec m_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec m_end = { .tv_sec = 2, .tv_nsec = 2 };
    struct timespec s_start = { .tv_sec = 3, .tv_nsec = 3 };
    struct timespec s_end = { .tv_sec = 4, .tv_nsec = 4 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_time_info(m_start, m_end, s_start, s_end, duration);
    EXPECT_EQ(ret, -1);
    tablength = orig_tablength;
}

// Edge-case: all time_print_element strings empty (should still succeed)
TEST(t_dlt_daemon_statistic_print_time_info, all_time_print_element_empty)
{
    extern char *time_print_element[];
    char *orig[4];
    char empty[] = "";
    for (int i = 0; i < 4; ++i) {
        orig[i] = time_print_element[i];
        time_print_element[i] = empty;
    }
    struct timespec m_start = { .tv_sec = 1, .tv_nsec = 1 };
    struct timespec m_end = { .tv_sec = 2, .tv_nsec = 2 };
    struct timespec s_start = { .tv_sec = 3, .tv_nsec = 3 };
    struct timespec s_end = { .tv_sec = 4, .tv_nsec = 4 };
    uint32_t duration = 1;
    int ret = dlt_daemon_statistic_print_time_info(m_start, m_end, s_start, s_end, duration);
    EXPECT_EQ(ret, 0);
    for (int i = 0; i < 4; ++i) {
        time_print_element[i] = orig[i];
    }
}

/* Test setting filename with NULL pointer and valid filename */
TEST(t_dlt_daemon_statistic_set_filename, normal)
{
    dlt_daemon_statistic_set_filename(NULL);
    ASSERT_STREQ(statistics_filename, "");

    char *filename = (char*)"dlt_log_statistics.csv";
    dlt_daemon_statistic_set_filename(filename);
    ASSERT_STREQ(statistics_filename, filename);

    dlt_daemon_statistic_set_filename(NULL);
    ASSERT_STREQ(statistics_filename, filename);
}

/* Test rejecting empty string as filename */
TEST(t_dlt_daemon_statistic_set_filename, empty_string)
{
    const char *filename = "/tmp/dlt_statistics.csv";
    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(filename));
    ASSERT_STREQ(statistics_filename, filename);
    EXPECT_EQ(-1, dlt_daemon_statistic_set_filename(STR_UNDEFINED));
    ASSERT_STREQ(statistics_filename, filename);
}

/* Test setting filename with exactly NAME_MAX characters - should succeed */
TEST(t_dlt_daemon_statistic_set_filename, boundary_max_length)
{
    // Create filename with exactly NAME_MAX characters (should succeed)
    char max_filename[NAME_MAX + 1];
    memset(max_filename, 'a', NAME_MAX);
    max_filename[NAME_MAX] = '\0';

    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(max_filename));
    ASSERT_STREQ(statistics_filename, max_filename);
}

/* Test filename exceeding NAME_MAX length - should fail and preserve previous filename */
TEST(t_dlt_daemon_statistic_set_filename, overflow_too_long)
{
    // Set a valid filename first
    const char *valid = "before_overflow.csv";
    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(valid));

    // Create filename with NAME_MAX + 1 characters (should fail)
    char too_long[NAME_MAX + 2];
    memset(too_long, 'b', NAME_MAX + 1);
    too_long[NAME_MAX + 1] = '\0';

    EXPECT_EQ(-1, dlt_daemon_statistic_set_filename(too_long));
    // Filename should remain unchanged after failure
    ASSERT_STREQ(statistics_filename, valid);
}

/* Test multiple consecutive filename changes - should update each time */
TEST(t_dlt_daemon_statistic_set_filename, multiple_calls)
{
    const char *file1 = "first.csv";
    const char *file2 = "second.csv";
    const char *file3 = "/var/log/third.csv";

    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(file1));
    ASSERT_STREQ(statistics_filename, file1);

    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(file2));
    ASSERT_STREQ(statistics_filename, file2);

    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(file3));
    ASSERT_STREQ(statistics_filename, file3);
}

/* Test recovery after error conditions - should accept valid filename after failures */
TEST(t_dlt_daemon_statistic_set_filename, error_recovery)
{
    const char *valid1 = "valid1.csv";
    const char *valid2 = "valid2.csv";

    // Set initial valid filename
    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(valid1));
    ASSERT_STREQ(statistics_filename, valid1);

    // Attempt NULL (should fail)
    EXPECT_EQ(-1, dlt_daemon_statistic_set_filename(NULL));
    ASSERT_STREQ(statistics_filename, valid1);

    // Attempt overflow (should fail)
    char overflow[NAME_MAX + 50];
    memset(overflow, 'x', NAME_MAX + 49);
    overflow[NAME_MAX + 49] = '\0';
    EXPECT_EQ(-1, dlt_daemon_statistic_set_filename(overflow));
    ASSERT_STREQ(statistics_filename, valid1);

    // Recovery with valid filename (should succeed)
    EXPECT_EQ(0, dlt_daemon_statistic_set_filename(valid2));
    ASSERT_STREQ(statistics_filename, valid2);
}

/* Test setting daemon internal pointers with valid and NULL values */
TEST(t_dlt_daemon_statistic_set_daemon_internal, normal)
{
    dlt_daemon_statistic_set_daemon_internal(NULL, NULL);
    ASSERT_TRUE(daemon_internal == NULL);
    ASSERT_TRUE(daemon_local_internal == NULL);

    DltDaemon daemon;
    DltDaemonLocal daemon_local;
    char *filename = (char*)"dlt_log_statistics.csv";

    if (clock_gettime(CLOCK_MONOTONIC, &daemon.stat_monotonic_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read monotonic time.\n", __func__);
    }

    if (clock_gettime(CLOCK_REALTIME, &daemon.stat_system_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read system time.\n", __func__);
    }

    daemon_local.flags.statisticMode = 1;
    strncpy(daemon_local.flags.statisticFilename, (char*)filename, sizeof(daemon_local.flags.statisticFilename) - 1);
    daemon_local.flags.statisticFilename[sizeof(daemon_local.flags.statisticFilename) - 1] = '\0';

    dlt_daemon_statistic_set_daemon_internal(&daemon, &daemon_local);
    ASSERT_TRUE(daemon_internal != NULL);
    ASSERT_TRUE(daemon_internal->stat_monotonic_time_start.tv_sec == daemon.stat_monotonic_time_start.tv_sec);
    ASSERT_TRUE(daemon_internal->stat_monotonic_time_start.tv_nsec == daemon.stat_monotonic_time_start.tv_nsec);
    ASSERT_TRUE(daemon_internal->stat_system_time_start.tv_sec == daemon.stat_system_time_start.tv_sec);
    ASSERT_TRUE(daemon_internal->stat_system_time_start.tv_nsec == daemon.stat_system_time_start.tv_nsec);

    ASSERT_TRUE(daemon_local_internal != NULL);
    ASSERT_TRUE(daemon_local_internal->flags.statisticMode == daemon_local.flags.statisticMode);
    ASSERT_STREQ(daemon_local_internal->flags.statisticFilename, daemon_local.flags.statisticFilename);
}

/* Test setting daemon internal pointers with valid and NULL values */
TEST(t_dlt_daemon_statistic_set_daemon_internal, normal_ptr_update)
{
    DltDaemon daemon;
    DltDaemonLocal daemon_local;
    dlt_daemon_statistic_set_daemon_internal(&daemon, &daemon_local);
    EXPECT_EQ(daemon_internal, &daemon);
    EXPECT_EQ(daemon_local_internal, &daemon_local);
}

/* Test resetting statistics for a single context with non-zero values */
TEST(t_dlt_daemon_statistic_reset, normal)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "TES";
    ID4 ctid = "CON";
    char desc[255] = "TEST dlt_daemon_context_add";
    DltDaemonContext *daecontext = NULL;
    DltDaemonApplication *app = NULL;
    char ecu[] = "ECU1";
    int fd = 42;
    int stat_cnt = 10;
    int stat_size = 20;

    EXPECT_EQ(0,
              dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE, DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                              DLT_DAEMON_RINGBUFFER_STEP_SIZE, DLT_RUNTIME_DEFAULT_DIRECTORY, DLT_LOG_INFO,
                              DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));
    EXPECT_EQ(DLT_RETURN_OK, strncmp(daemon.ecuid, daemon.user_list[0].ecu, DLT_ID_SIZE));
    app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    daecontext = dlt_daemon_context_add(&daemon, apid, ctid, DLT_LOG_DEFAULT,
                                        DLT_TRACE_STATUS_DEFAULT, 0, 0, desc, ecu, 0);

    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        daecontext->log_level_statistics[i].stat_cnt = stat_cnt;
        daecontext->log_level_statistics[i].stat_size = stat_size;
    }

    EXPECT_STREQ(apid, daecontext->apid);
    EXPECT_STREQ(ctid, daecontext->ctid);
    EXPECT_STREQ(desc, daecontext->context_description);
    EXPECT_EQ(DLT_LOG_DEFAULT, daecontext->log_level);
    EXPECT_EQ(DLT_TRACE_STATUS_DEFAULT, daecontext->trace_status);

    DltDaemonRegisteredUsers *user_list;
    user_list = dlt_daemon_find_users_list(&daemon, &ecu[0], 0);
    EXPECT_NE(user_list, nullptr);

    if (user_list->num_applications > 0) {
        for (int app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
            DltDaemonApplication *application = &user_list->applications[app_idx];
            DltDaemonContext *context;
            int i, offset_base;

            /* Calculate start offset within contexts[] */
            offset_base = 0;

            for (i = 0; i < (application - (user_list->applications)); i++) {
                offset_base += user_list->applications[i].num_contexts;
            }

            for (i = (application->num_contexts) - 1; i >= 0; i--) {
                context = &(user_list->contexts[offset_base + i]);
                for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
                    EXPECT_EQ(context->log_level_statistics[i].stat_cnt, daecontext->log_level_statistics[i].stat_cnt);
                    EXPECT_EQ(context->log_level_statistics[i].stat_size, daecontext->log_level_statistics[i].stat_size);
                }
            }
        }
    }

    dlt_daemon_statistic_reset(user_list);
    if (user_list->num_applications > 0) {
        for (int app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
            DltDaemonApplication *application = &user_list->applications[app_idx];
            DltDaemonContext *context;
            int i, offset_base;

            /* Calculate start offset within contexts[] */
            offset_base = 0;

            for (i = 0; i < (application - (user_list->applications)); i++) {
                offset_base += user_list->applications[i].num_contexts;
            }

            for (i = (application->num_contexts) - 1; i >= 0; i--) {
                context = &(user_list->contexts[offset_base + i]);
                for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
                    EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
                    EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
                }
            }
        }
    }

    EXPECT_LE(0, dlt_daemon_context_del(&daemon, daecontext, ecu, 0));
    EXPECT_LE(0, dlt_daemon_application_del(&daemon, app, ecu, 0));
    EXPECT_LE(0, dlt_daemon_contexts_clear(&daemon, ecu, 0));
    EXPECT_LE(0, dlt_daemon_applications_clear(&daemon, ecu, 0));
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset with NULL user_list pointer - should return error */
TEST(t_dlt_daemon_statistic_reset, null_pointer)
{
    EXPECT_EQ(-1, dlt_daemon_statistic_reset(NULL));
}

/* Test reset with empty user list (no applications) - should return error */
TEST(t_dlt_daemon_statistic_reset, empty_user_list)
{
    DltDaemonRegisteredUsers user_list;
    memset(&user_list, 0, sizeof(DltDaemonRegisteredUsers));

    user_list.num_applications = 0;
    user_list.applications = NULL;
    user_list.contexts = NULL;

    EXPECT_EQ(-1, dlt_daemon_statistic_reset(&user_list));
}

/* Test reset with NULL applications pointer - should return error */
TEST(t_dlt_daemon_statistic_reset, null_applications_pointer)
{
    DltDaemonRegisteredUsers user_list;
    memset(&user_list, 0, sizeof(DltDaemonRegisteredUsers));

    user_list.num_applications = 1;
    user_list.applications = NULL;
    user_list.contexts = NULL;

    EXPECT_EQ(-1, dlt_daemon_statistic_reset(&user_list));
}

/* Test reset with NULL contexts pointer - should return error */
TEST(t_dlt_daemon_statistic_reset, null_contexts_pointer)
{
    DltDaemon daemon;
    DltGateway gateway;
    char ecu[] = "ECU1";

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    ASSERT_NE(user_list, nullptr);

    // Intentionally set contexts to NULL
    user_list->contexts = NULL;

    EXPECT_EQ(-1, dlt_daemon_statistic_reset(user_list));

    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset with single application and single context - should zero all statistics */
TEST(t_dlt_daemon_statistic_reset, single_app_single_context)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "APP";
    ID4 ctid = "CTX";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0,
                                                             (char*)"Test app", fd, ecu, 0);
    ASSERT_NE(app, nullptr);

    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                         DLT_LOG_DEFAULT,
                                                         DLT_TRACE_STATUS_DEFAULT,
                                                         0, 0, (char*)"Test ctx", ecu, 0);
    ASSERT_NE(context, nullptr);

    // Set statistics values
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        context->log_level_statistics[i].stat_cnt = 100;
        context->log_level_statistics[i].stat_size = 5000;
    }

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    ASSERT_NE(user_list, nullptr);

    // Reset statistics
    EXPECT_EQ(0, dlt_daemon_statistic_reset(user_list));

    // Verify all statistics are reset
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
        EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
    }

    dlt_daemon_context_del(&daemon, context, ecu, 0);
    dlt_daemon_application_del(&daemon, app, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset with multiple applications and contexts - should zero all statistics */
TEST(t_dlt_daemon_statistic_reset, multiple_apps_multiple_contexts)
{
    DltDaemon daemon;
    DltGateway gateway;
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    // Create multiple applications with contexts
    const int num_apps = 3;
    const int contexts_per_app = 3;

    for (int app_idx = 0; app_idx < num_apps; app_idx++) {
        char apid[5];
        snprintf(apid, sizeof(apid), "AP%d", app_idx);

        DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0,
                                                                 (char*)"Test app", fd, ecu, 0);
        ASSERT_NE(app, nullptr);

        for (int ctx_idx = 0; ctx_idx < contexts_per_app; ctx_idx++) {
            char ctid[5];
            snprintf(ctid, sizeof(ctid), "C%d%d", app_idx, ctx_idx);

            DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                                 DLT_LOG_DEFAULT,
                                                                 DLT_TRACE_STATUS_DEFAULT,
                                                                 0, 0, (char*)"Test ctx", ecu, 0);
            ASSERT_NE(context, nullptr);

            // Set different statistics for each context
            for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
                context->log_level_statistics[i].stat_cnt = (app_idx + 1) * (ctx_idx + 1) * 10;
                context->log_level_statistics[i].stat_size = (app_idx + 1) * (ctx_idx + 1) * 1000;
            }
        }
    }

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    ASSERT_NE(user_list, nullptr);

    // Reset all statistics
    EXPECT_EQ(0, dlt_daemon_statistic_reset(user_list));

    // Verify all contexts have reset statistics
    int offset = 0;
    for (int app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
        DltDaemonApplication *app = &user_list->applications[app_idx];

        for (int ctx_idx = 0; ctx_idx < app->num_contexts; ctx_idx++) {
            DltDaemonContext *context = &user_list->contexts[offset + ctx_idx];

            for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
                EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
                EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
            }
        }
        offset += app->num_contexts;
    }

    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset with statistics at maximum UINT32_MAX values - should zero them */
TEST(t_dlt_daemon_statistic_reset, max_statistics_values)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "MAX";
    ID4 ctid = "VAL";
    char desc[255] = "Max values test";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                         DLT_LOG_DEFAULT,
                                                         DLT_TRACE_STATUS_DEFAULT,
                                                         0, 0, desc, ecu, 0);

    // Set maximum uint32_t values
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        context->log_level_statistics[i].stat_cnt = UINT32_MAX;
        context->log_level_statistics[i].stat_size = UINT32_MAX;
    }

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_statistic_reset(user_list));

    // Verify reset
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
        EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
    }

    dlt_daemon_context_del(&daemon, context, ecu, 0);
    dlt_daemon_application_del(&daemon, app, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset when statistics are already zero - should remain zero */
TEST(t_dlt_daemon_statistic_reset, zero_statistics)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "ZER";
    ID4 ctid = "CTX";
    char desc[255] = "Zero test";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                         DLT_LOG_DEFAULT,
                                                         DLT_TRACE_STATUS_DEFAULT,
                                                         0, 0, desc, ecu, 0);

    // Statistics are already 0 by default
    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_statistic_reset(user_list));

    // Verify still 0
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
        EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
    }

    dlt_daemon_context_del(&daemon, context, ecu, 0);
    dlt_daemon_application_del(&daemon, app, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test multiple consecutive reset operations - should work repeatedly */
TEST(t_dlt_daemon_statistic_reset, consecutive_resets)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "CON";
    ID4 ctid = "SEC";
    char desc[255] = "Consecutive test";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                         DLT_LOG_DEFAULT,
                                                         DLT_TRACE_STATUS_DEFAULT,
                                                         0, 0, desc, ecu, 0);

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);

    // Reset multiple times
    for (int reset_count = 0; reset_count < 5; reset_count++) {
        // Set statistics
        for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
            context->log_level_statistics[i].stat_cnt = (reset_count + 1) * 50;
            context->log_level_statistics[i].stat_size = (reset_count + 1) * 2000;
        }

        // Reset
        EXPECT_EQ(0, dlt_daemon_statistic_reset(user_list));

        // Verify reset
        for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
            EXPECT_EQ(context->log_level_statistics[i].stat_cnt, 0);
            EXPECT_EQ(context->log_level_statistics[i].stat_size, 0);
        }
    }

    dlt_daemon_context_del(&daemon, context, ecu, 0);
    dlt_daemon_application_del(&daemon, app, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test reset with application having no contexts - should return error */
TEST(t_dlt_daemon_statistic_reset, app_with_no_contexts)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "NOC";
    char desc[255] = "No contexts";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    // Add application without contexts
    DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    ASSERT_NE(app, nullptr);

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);

    EXPECT_EQ(-1, dlt_daemon_statistic_reset(user_list));

    dlt_daemon_application_del(&daemon, app, ecu, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Helper function to check if a specific string exists in a file returns 1 if found, 0 otherwise */
int string_existing_in_file(char *filepath, char *str) {
    std::ifstream file(filepath);
    std::string line_str;
    std::string input_str(str);
    int found = 0;

    if (file.is_open()) {
        while (!file.eof()) {
            getline(file, line_str);
            if (line_str.size() == 0) {
                continue;
            }
            if (line_str.find(input_str) != std::string::npos) {
                found = 1;
            }
        }
        file.close();
    }
    return found;
}

/* Test printing statistics to CSV file with valid data and filename */
TEST(t_dlt_daemon_statistic_print, normal)
{
    DltDaemonLocal daemon_local;
    DltFilterConfiguration current;
    daemon_local.pFilter.current = &current;
    DltGatewayConnection connections;
    DltConnection connections1;
    DltReceiver receiver1;
    DltPassiveControlMessage p_control_msgs;
    memset(&daemon_local,0, sizeof(DltDaemonLocal));
    memset(&current, 0, sizeof(DltFilterConfiguration));
    memset(&connections, 0, sizeof(DltGatewayConnection));
    memset(&p_control_msgs,0, sizeof(DltPassiveControlMessage));

    daemon_local.pFilter.current = &current;
    daemon_local.pGateway.connections = &connections;
    daemon_local.pEvent.connections = &connections1;
    daemon_local.pGateway.connections->p_control_msgs = &p_control_msgs;
    daemon_local.pEvent.connections->next = NULL;
    daemon_local.pEvent.pfd = 0;
    daemon_local.pEvent.nfds = 0;
    daemon_local.pEvent.max_nfds = 0;
    daemon_local.pEvent.connections->receiver = &receiver1;
    memset(daemon_local.flags.gatewayConfigFile, 0, DLT_DAEMON_FLAG_MAX);

    daemon_local.pGateway.connections->p_control_msgs->id = DLT_SERVICE_ID_GET_LOG_INFO;
    daemon_local.pGateway.connections->p_control_msgs->type = CONTROL_MESSAGE_ON_DEMAND;

    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "TES";
    ID4 ctid = "CON";
    char desc[255] = "TEST dlt_daemon_context_add";
    DltDaemonContext *daecontext = NULL;
    DltDaemonApplication *app = NULL;
    char ecu[] = "ECU1";
    int fd = 42;
    int duration = 5;

    int stat_cnt = 10;
    int stat_size = 20;
    char *filename = (char*)"dlt_log_statistics.csv";

    daemon_local.flags.statisticMode = 1;
    daemon_local.flags.vflag = 1;
    strncpy(daemon_local.flags.statisticFilename, (char*)filename, sizeof(daemon_local.flags.statisticFilename) - 1);
    daemon_local.flags.statisticFilename[sizeof(daemon_local.flags.statisticFilename) - 1] = '\0';
    dlt_message_init(&daemon_local.msg, 0);
    daemon_local.msg.standardheader = (DltStandardHeader *)(daemon_local.msg.headerbuffer + sizeof(DltStorageHeader));
    daemon_local.msg.standardheader->htyp = DLT_HTYP_PROTOCOL_VERSION1|DLT_HTYP_UEH;
    daemon_local.msg.standardheader->mcnt = 0;
    dlt_message_set_extraparameters(&daemon_local.msg, 0);
    daemon_local.msg.extendedheader = (DltExtendedHeader *)(daemon_local.msg.headerbuffer +
                        sizeof(DltStorageHeader) +
                        sizeof(DltStandardHeader) +
                        DLT_STANDARD_HEADER_EXTRA_SIZE(daemon_local.msg.standardheader->htyp));
    dlt_set_id(daemon_local.msg.extendedheader->apid, apid);
    dlt_set_id(daemon_local.msg.extendedheader->ctid, ctid);

    if (clock_gettime(CLOCK_MONOTONIC, &daemon.stat_monotonic_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read monotonic time.\n", __func__);
    }
    struct timespec stat_monotonic_time_end;
    if (clock_gettime(CLOCK_MONOTONIC, &stat_monotonic_time_end) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read monotonic time.\n", __func__);
    }

    if (clock_gettime(CLOCK_REALTIME, &daemon.stat_system_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read system time.\n", __func__);
    }
    struct timespec stat_system_time_end;
    if (clock_gettime(CLOCK_REALTIME, &stat_system_time_end) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read system time.\n", __func__);
    }

    dlt_daemon_statistic_set_daemon_internal(&daemon, &daemon_local);

    stat_monotonic_time_end.tv_sec += duration;
    stat_system_time_end.tv_sec += duration;

    /* Normal Use-Case */
    EXPECT_EQ(0,
            dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE, DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                            DLT_DAEMON_RINGBUFFER_STEP_SIZE, DLT_RUNTIME_DEFAULT_DIRECTORY, DLT_LOG_INFO,
                            DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));
    EXPECT_EQ(DLT_RETURN_OK, strncmp(daemon.ecuid, daemon.user_list[0].ecu, DLT_ID_SIZE));
    app = dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    daecontext = dlt_daemon_context_add(&daemon, apid, ctid, DLT_LOG_DEFAULT,
                                        DLT_TRACE_STATUS_DEFAULT, 0, 0, desc, ecu, 0);

    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        daecontext->log_level_statistics[i].stat_cnt = stat_cnt;
        daecontext->log_level_statistics[i].stat_size = stat_size;
    }

    EXPECT_STREQ(apid, daecontext->apid);
    EXPECT_STREQ(ctid, daecontext->ctid);
    EXPECT_STREQ(desc, daecontext->context_description);
    EXPECT_EQ(DLT_LOG_DEFAULT, daecontext->log_level);
    EXPECT_EQ(DLT_TRACE_STATUS_DEFAULT, daecontext->trace_status);

    DltDaemonRegisteredUsers *user_list;
    user_list = dlt_daemon_find_users_list(&daemon, &ecu[0], 0);
    EXPECT_NE(user_list, nullptr);

    dlt_daemon_statistic_set_filename(filename);
    ASSERT_STREQ(statistics_filename, filename);

    dlt_daemon_statistic_print(user_list, daemon.stat_monotonic_time_start, stat_monotonic_time_end,
                                daemon.stat_system_time_start, stat_system_time_end);

    struct stat buffer;
    EXPECT_EQ(0, stat(filename, &buffer));

    EXPECT_EQ(1, string_existing_in_file(filename, ecu));
    EXPECT_EQ(1, string_existing_in_file(filename, apid));
    EXPECT_EQ(1, string_existing_in_file(filename, ctid));

    char subline[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH + 1] = { 0 };
    snprintf(subline, sizeof(subline), "%5u.%06u", (unsigned int)daemon.stat_monotonic_time_start.tv_sec, (unsigned int)daemon.stat_monotonic_time_start.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u", (unsigned int)stat_monotonic_time_end.tv_sec, (unsigned int)stat_monotonic_time_end.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u", (unsigned int)daemon.stat_system_time_start.tv_sec, (unsigned int)daemon.stat_system_time_start.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u", (unsigned int)stat_system_time_end.tv_sec, (unsigned int)stat_system_time_end.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    EXPECT_LE(0, dlt_message_free(&daemon_local.msg, 0));
    EXPECT_LE(0, dlt_daemon_context_del(&daemon, daecontext, ecu, 0));
    EXPECT_LE(0, dlt_daemon_application_del(&daemon, app, ecu, 0));
    EXPECT_LE(0, dlt_daemon_contexts_clear(&daemon, ecu, 0));
    EXPECT_LE(0, dlt_daemon_applications_clear(&daemon, ecu, 0));
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));

    if (stat(filename, &buffer) == 0) {
        char cmd[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH + 1] = { 0 };
        snprintf(cmd, sizeof(cmd), "rm -rf %s", filename);
        EXPECT_EQ(DLT_RETURN_OK, system(cmd));
    }
}

/* Test printing statistics with multiple contexts across applications */
TEST(t_dlt_daemon_statistic_print, multiple_contexts)
{
    DltDaemonLocal daemon_local;
    DltFilterConfiguration current;
    daemon_local.pFilter.current = &current;
    DltGatewayConnection connections;
    DltConnection connections1;
    DltReceiver receiver1;
    DltPassiveControlMessage p_control_msgs;
    memset(&daemon_local, 0, sizeof(DltDaemonLocal));
    memset(&current, 0, sizeof(DltFilterConfiguration));
    memset(&connections, 0, sizeof(DltGatewayConnection));
    memset(&p_control_msgs, 0, sizeof(DltPassiveControlMessage));

    daemon_local.pFilter.current = &current;
    daemon_local.pGateway.connections = &connections;
    daemon_local.pEvent.connections = &connections1;
    daemon_local.pGateway.connections->p_control_msgs = &p_control_msgs;
    daemon_local.pEvent.connections->next = NULL;
    daemon_local.pEvent.pfd = 0;
    daemon_local.pEvent.nfds = 0;
    daemon_local.pEvent.max_nfds = 0;
    daemon_local.pEvent.connections->receiver = &receiver1;
    memset(daemon_local.flags.gatewayConfigFile, 0, DLT_DAEMON_FLAG_MAX);

    daemon_local.pGateway.connections->p_control_msgs->id = DLT_SERVICE_ID_GET_LOG_INFO;
    daemon_local.pGateway.connections->p_control_msgs->type = CONTROL_MESSAGE_ON_DEMAND;

    DltDaemon daemon;
    DltGateway gateway;
    char ecu[] = "ECU1";
    int fd = 42;
    int duration = 10;

    char *filename = (char*)"/tmp/dlt_log_statistics_multi.csv";

    daemon_local.flags.statisticMode = 1;
    daemon_local.flags.vflag = 1;
    strncpy(daemon_local.flags.statisticFilename, filename, sizeof(daemon_local.flags.statisticFilename) - 1);
    daemon_local.flags.statisticFilename[sizeof(daemon_local.flags.statisticFilename) - 1] = '\0';

    // Initialize message structure with first context IDs
    dlt_message_init(&daemon_local.msg, 0);
    daemon_local.msg.standardheader = (DltStandardHeader *)(daemon_local.msg.headerbuffer + sizeof(DltStorageHeader));
    daemon_local.msg.standardheader->htyp = DLT_HTYP_PROTOCOL_VERSION1 | DLT_HTYP_UEH;
    daemon_local.msg.standardheader->mcnt = 0;
    dlt_message_set_extraparameters(&daemon_local.msg, 0);
    daemon_local.msg.extendedheader = (DltExtendedHeader *)(daemon_local.msg.headerbuffer +
                         sizeof(DltStorageHeader) +
                         sizeof(DltStandardHeader) +
                         DLT_STANDARD_HEADER_EXTRA_SIZE(daemon_local.msg.standardheader->htyp));
    dlt_set_id(daemon_local.msg.extendedheader->apid, "AP0");
    dlt_set_id(daemon_local.msg.extendedheader->ctid, "C00");

    if (clock_gettime(CLOCK_MONOTONIC, &daemon.stat_monotonic_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read monotonic time.\n", __func__);
    }
    struct timespec stat_monotonic_time_end;
    if (clock_gettime(CLOCK_MONOTONIC, &stat_monotonic_time_end) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read monotonic time.\n", __func__);
    }

    if (clock_gettime(CLOCK_REALTIME, &daemon.stat_system_time_start) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read system time.\n", __func__);
    }
    struct timespec stat_system_time_end;
    if (clock_gettime(CLOCK_REALTIME, &stat_system_time_end) == -1) {
        dlt_vlog(LOG_ERR, "%s: Cannot read system time.\n", __func__);
    }

    dlt_daemon_statistic_set_daemon_internal(&daemon, &daemon_local);

    stat_monotonic_time_end.tv_sec += duration;
    stat_system_time_end.tv_sec += duration;

    //Initialize daemon
    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE, DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE, DLT_RUNTIME_DEFAULT_DIRECTORY, DLT_LOG_INFO,
                                  DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));
    EXPECT_EQ(DLT_RETURN_OK, strncmp(daemon.ecuid, daemon.user_list[0].ecu, DLT_ID_SIZE));

    // Create 2 applications with 2 contexts each
    const int num_apps = 2;
    const int contexts_per_app = 2;

    for (int app_idx = 0; app_idx < num_apps; app_idx++) {
        char apid[5];
        snprintf(apid, sizeof(apid), "AP%d", app_idx);
        char app_desc[255];
        snprintf(app_desc, sizeof(app_desc), "Test Application %d", app_idx);

        DltDaemonApplication *app = dlt_daemon_application_add(&daemon, apid, 0, app_desc, fd, ecu, 0);
        ASSERT_NE(app, nullptr);

        for (int ctx_idx = 0; ctx_idx < contexts_per_app; ctx_idx++) {
            char ctid[5];
            snprintf(ctid, sizeof(ctid), "C%d%d", app_idx, ctx_idx);
            char ctx_desc[255];
            snprintf(ctx_desc, sizeof(ctx_desc), "Test Context %d-%d", app_idx, ctx_idx);

            DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid, DLT_LOG_DEFAULT,
                                                                DLT_TRACE_STATUS_DEFAULT, 0, 0, ctx_desc, ecu, 0);
            ASSERT_NE(context, nullptr);

            // Set different statistics for each context to verify individual tracking
            for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
                context->log_level_statistics[i].stat_cnt = (app_idx + 1) * (ctx_idx + 1) * 10 * (i + 1);
                context->log_level_statistics[i].stat_size = (app_idx + 1) * (ctx_idx + 1) * 500 * (i + 1);
            }

            EXPECT_STREQ(apid, context->apid);
            EXPECT_STREQ(ctid, context->ctid);
            EXPECT_STREQ(ctx_desc, context->context_description);
            EXPECT_EQ(DLT_LOG_DEFAULT, context->log_level);
            EXPECT_EQ(DLT_TRACE_STATUS_DEFAULT, context->trace_status);
        }
    }

    DltDaemonRegisteredUsers *user_list;
    user_list = dlt_daemon_find_users_list(&daemon, &ecu[0], 0);
    EXPECT_NE(user_list, nullptr);
    EXPECT_EQ(num_apps, user_list->num_applications);

    dlt_daemon_statistic_set_filename(filename);
    ASSERT_STREQ(statistics_filename, filename);

    // Print statistics
    EXPECT_EQ(0, dlt_daemon_statistic_print(user_list, daemon.stat_monotonic_time_start, stat_monotonic_time_end,
                                             daemon.stat_system_time_start, stat_system_time_end));

    struct stat buffer;
    EXPECT_EQ(0, stat(filename, &buffer));

    // Verify ECU ID is in file
    EXPECT_EQ(1, string_existing_in_file(filename, ecu));

    // Verify all applications are in file
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"AP0"));
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"AP1"));

    // Verify all contexts are in file
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"C00"));
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"C01"));
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"C10"));
    EXPECT_EQ(1, string_existing_in_file(filename, (char*)"C11"));

    // Verify time information
    char subline[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH + 1] = { 0 };
    snprintf(subline, sizeof(subline), "%5u.%06u",
             (unsigned int)daemon.stat_monotonic_time_start.tv_sec,
             (unsigned int)daemon.stat_monotonic_time_start.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u",
             (unsigned int)stat_monotonic_time_end.tv_sec,
             (unsigned int)stat_monotonic_time_end.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u",
             (unsigned int)daemon.stat_system_time_start.tv_sec,
             (unsigned int)daemon.stat_system_time_start.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    snprintf(subline, sizeof(subline), "%5u.%06u",
             (unsigned int)stat_system_time_end.tv_sec,
             (unsigned int)stat_system_time_end.tv_nsec/NSEC_TO_USEC);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    // Verify duration
    snprintf(subline, sizeof(subline), "%d", duration);
    EXPECT_EQ(1, string_existing_in_file(filename, subline));

    // Cleanup
    EXPECT_LE(0, dlt_message_free(&daemon_local.msg, 0));

    // Clean up all contexts and applications
    int offset = 0;
    for (int app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
        DltDaemonApplication *app = &user_list->applications[app_idx];

        for (int ctx_idx = app->num_contexts - 1; ctx_idx >= 0; ctx_idx--) {
            DltDaemonContext *context = &user_list->contexts[offset + ctx_idx];
            EXPECT_LE(0, dlt_daemon_context_del(&daemon, context, ecu, 0));
        }
        offset += app->num_contexts;
        EXPECT_LE(0, dlt_daemon_application_del(&daemon, app, ecu, 0));
    }

    EXPECT_LE(0, dlt_daemon_contexts_clear(&daemon, ecu, 0));
    EXPECT_LE(0, dlt_daemon_applications_clear(&daemon, ecu, 0));
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));

    // Remove test file
    if (stat(filename, &buffer) == 0) {
        char cmd[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH + 1] = { 0 };
        snprintf(cmd, sizeof(cmd), "rm -rf %s", filename);
        EXPECT_EQ(DLT_RETURN_OK, system(cmd));
    }
}

/* Test print with NULL user_list pointer - should return error */
TEST(t_dlt_daemon_statistic_print, null_user_list)
{
    struct timespec m_start, m_end, s_start, s_end;

    clock_gettime(CLOCK_MONOTONIC, &m_start);
    clock_gettime(CLOCK_MONOTONIC, &m_end);
    clock_gettime(CLOCK_REALTIME, &s_start);
    clock_gettime(CLOCK_REALTIME, &s_end);

    m_end.tv_sec += 5;
    s_end.tv_sec += 5;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(NULL, m_start, m_end, s_start, s_end));
}

/* Test print with empty user list (no applications) - should return error */
TEST(t_dlt_daemon_statistic_print, empty_user_list)
{
    DltDaemonRegisteredUsers user_list;
    memset(&user_list, 0, sizeof(DltDaemonRegisteredUsers));

    user_list.num_applications = 0;
    user_list.applications = NULL;
    user_list.contexts = NULL;

    struct timespec m_start, m_end, s_start, s_end;
    clock_gettime(CLOCK_MONOTONIC, &m_start);
    clock_gettime(CLOCK_MONOTONIC, &m_end);
    clock_gettime(CLOCK_REALTIME, &s_start);
    clock_gettime(CLOCK_REALTIME, &s_end);

    m_end.tv_sec += 5;
    s_end.tv_sec += 5;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(&user_list, m_start, m_end, s_start, s_end));
}

/* Test print with NULL applications pointer - should return error */
TEST(t_dlt_daemon_statistic_print, null_applications)
{
    DltDaemonRegisteredUsers user_list;
    memset(&user_list, 0, sizeof(DltDaemonRegisteredUsers));

    user_list.num_applications = 1;
    user_list.applications = NULL;
    user_list.contexts = NULL;

    struct timespec m_start, m_end, s_start, s_end;
    clock_gettime(CLOCK_MONOTONIC, &m_start);
    clock_gettime(CLOCK_MONOTONIC, &m_end);
    clock_gettime(CLOCK_REALTIME, &s_start);
    clock_gettime(CLOCK_REALTIME, &s_end);

    m_end.tv_sec += 5;
    s_end.tv_sec += 5;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(&user_list, m_start, m_end, s_start, s_end));
}

/* Test print with NULL contexts pointer - should return error */
TEST(t_dlt_daemon_statistic_print, null_contexts)
{
    DltDaemon daemon;
    DltGateway gateway;
    char ecu[] = "ECU1";

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);
    ASSERT_NE(user_list, nullptr);

    user_list->contexts = NULL;

    struct timespec m_start, m_end, s_start, s_end;
    clock_gettime(CLOCK_MONOTONIC, &m_start);
    clock_gettime(CLOCK_MONOTONIC, &m_end);
    clock_gettime(CLOCK_REALTIME, &s_start);
    clock_gettime(CLOCK_REALTIME, &s_end);

    m_end.tv_sec += 5;
    s_end.tv_sec += 5;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(user_list, m_start, m_end, s_start, s_end));

    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test print with end time before start time - should return error */
TEST(t_dlt_daemon_statistic_print, invalid_time_range)
{
    DltDaemon daemon;
    DltGateway gateway;
    ID4 apid = "APP";
    ID4 ctid = "CTX";
    char desc[255] = "Test";
    char ecu[] = "ECU1";
    int fd = 42;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    dlt_daemon_context_add(&daemon, apid, ctid, DLT_LOG_DEFAULT,
                           DLT_TRACE_STATUS_DEFAULT, 0, 0, desc, ecu, 0);

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);

    struct timespec m_start, m_end, s_start, s_end;
    clock_gettime(CLOCK_MONOTONIC, &m_start);
    clock_gettime(CLOCK_MONOTONIC, &m_end);
    clock_gettime(CLOCK_REALTIME, &s_start);
    clock_gettime(CLOCK_REALTIME, &s_end);

    // Set end time before start time (invalid)
    m_end.tv_sec = m_start.tv_sec - 10;
    s_end.tv_sec = s_start.tv_sec - 10;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(user_list, m_start, m_end, s_start, s_end));

    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test print with zero duration (same start and end time) - should return error */
TEST(t_dlt_daemon_statistic_print, zero_duration)
{
    DltDaemon daemon;
    DltDaemonLocal daemon_local;
    DltGateway gateway;
    ID4 apid = "APP";
    ID4 ctid = "CTX";
    char desc[255] = "Test";
    char ecu[] = "ECU1";
    char filename[] = "/tmp/test_zero_duration.csv";
    int fd = 42;

    memset(&daemon_local, 0, sizeof(DltDaemonLocal));
    daemon_local.flags.statisticMode = 1;
    daemon_local.flags.vflag = 1;

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    dlt_daemon_application_add(&daemon, apid, 0, desc, fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                        DLT_LOG_DEFAULT,
                                                        DLT_TRACE_STATUS_DEFAULT,
                                                        0, 0, desc, ecu, 0);

    // Set some statistics
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        context->log_level_statistics[i].stat_cnt = 100;
        context->log_level_statistics[i].stat_size = 1000;
    }

    dlt_daemon_statistic_set_daemon_internal(&daemon, &daemon_local);
    dlt_daemon_statistic_set_filename(filename);

    DltDaemonRegisteredUsers *user_list = dlt_daemon_find_users_list(&daemon, ecu, 0);

    struct timespec m_start, m_end, s_start, s_end;
    clock_gettime(CLOCK_MONOTONIC, &m_start);
    m_end = m_start; // Same time (zero duration)
    clock_gettime(CLOCK_REALTIME, &s_start);
    s_end = s_start;

    EXPECT_EQ(-1, dlt_daemon_statistic_print(user_list, m_start, m_end, s_start, s_end));

    // Cleanup
    remove(filename);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

TEST(t_dlt_daemon_process_message_log_statistic, non_log_message_ignored)
{
    DltDaemon daemon;
    DltGateway gateway;
    DltDaemonLocal daemon_local;
    ID4 apid = "APP";
    ID4 ctid = "CTX";
    char ecu[] = "ECU1";
    int fd = 42;

    memset(&daemon, 0, sizeof(DltDaemon));
    memset(&daemon_local, 0, sizeof(DltDaemonLocal));

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    daemon_local.flags.statisticMode = DLT_DAEMON_STATISTIC_ENABLED;

    dlt_daemon_application_add(&daemon, apid, 0, (char*)"Test app", fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                        DLT_LOG_DEFAULT,
                                                        DLT_TRACE_STATUS_DEFAULT,
                                                        0, 0, (char*)"Test ctx", ecu, 0);
    ASSERT_NE(context, nullptr);

    context->log_level_statistics[DLT_LOG_INFO].stat_cnt = 0;

    dlt_message_init(&daemon_local.msg, 0);
    daemon_local.msg.standardheader = (DltStandardHeader*)calloc(1, sizeof(DltStandardHeader));
    daemon_local.msg.extendedheader = (DltExtendedHeader*)calloc(1, sizeof(DltExtendedHeader));
    ASSERT_NE(daemon_local.msg.standardheader, nullptr);
    ASSERT_NE(daemon_local.msg.extendedheader, nullptr);

    daemon_local.msg.standardheader->htyp = DLT_HTYP_PROTOCOL_VERSION1 | DLT_HTYP_UEH;
    daemon_local.msg.standardheader->len = DLT_HTOBE_16(100);
    daemon_local.msg.datasize = 50;

    dlt_set_id(daemon_local.msg.extendedheader->apid, apid);
    dlt_set_id(daemon_local.msg.extendedheader->ctid, ctid);
    // Set as CONTROL message, not LOG
    daemon_local.msg.extendedheader->msin = DLT_TYPE_CONTROL << DLT_MSIN_MSTP_SHIFT;

    dlt_daemon_process_message_log_statistic(&daemon, &daemon_local, 0);

    // Verify no statistics updated (control messages don't count)
    EXPECT_EQ(0, context->log_level_statistics[DLT_LOG_INFO].stat_cnt);

    free(daemon_local.msg.standardheader);
    free(daemon_local.msg.extendedheader);
    dlt_message_free(&daemon_local.msg, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Test that counter saturates at UINT32_MAX without overflow */
TEST(t_dlt_daemon_process_message_log_statistic, counter_at_max_saturates)
{
    DltDaemon daemon;
    DltGateway gateway;
    DltDaemonLocal daemon_local;
    ID4 apid = "APP";
    ID4 ctid = "CTX";
    char ecu[] = "ECU1";
    int fd = 42;

    memset(&daemon, 0, sizeof(DltDaemon));
    memset(&daemon_local, 0, sizeof(DltDaemonLocal));

    EXPECT_EQ(0, dlt_daemon_init(&daemon, DLT_DAEMON_RINGBUFFER_MIN_SIZE,
                                  DLT_DAEMON_RINGBUFFER_MAX_SIZE,
                                  DLT_DAEMON_RINGBUFFER_STEP_SIZE,
                                  DLT_RUNTIME_DEFAULT_DIRECTORY,
                                  DLT_LOG_INFO, DLT_TRACE_STATUS_OFF, 0, 0));
    dlt_set_id(daemon.ecuid, ecu);
    EXPECT_EQ(0, dlt_daemon_init_user_information(&daemon, &gateway, 0, 0));

    daemon_local.flags.statisticMode = DLT_DAEMON_STATISTIC_ENABLED;

    dlt_daemon_application_add(&daemon, apid, 0, (char*)"Test app", fd, ecu, 0);
    DltDaemonContext *context = dlt_daemon_context_add(&daemon, apid, ctid,
                                                        DLT_LOG_DEFAULT,
                                                        DLT_TRACE_STATUS_DEFAULT,
                                                        0, 0, (char*)"Test ctx", ecu, 0);
    ASSERT_NE(context, nullptr);

    // Set counter to UINT32_MAX (maximum value)
    context->log_level_statistics[DLT_LOG_INFO].stat_cnt = UINT32_MAX;
    context->log_level_statistics[DLT_LOG_INFO].stat_size = UINT32_MAX - 500;

    // Store initial size for verification
    uint32_t initial_size = context->log_level_statistics[DLT_LOG_INFO].stat_size;

    // Prepare message
    dlt_message_init(&daemon_local.msg, 0);
    daemon_local.msg.standardheader = (DltStandardHeader*)calloc(1, sizeof(DltStandardHeader));
    daemon_local.msg.extendedheader = (DltExtendedHeader*)calloc(1, sizeof(DltExtendedHeader));
    ASSERT_NE(daemon_local.msg.standardheader, nullptr);
    ASSERT_NE(daemon_local.msg.extendedheader, nullptr);

    daemon_local.msg.standardheader->htyp = DLT_HTYP_PROTOCOL_VERSION1 | DLT_HTYP_UEH;
    daemon_local.msg.standardheader->len = DLT_HTOBE_16(100);
    daemon_local.msg.datasize = 50;

    dlt_set_id(daemon_local.msg.extendedheader->apid, apid);
    dlt_set_id(daemon_local.msg.extendedheader->ctid, ctid);
    daemon_local.msg.extendedheader->msin = DLT_TYPE_LOG << DLT_MSIN_MSTP_SHIFT |
                                (DLT_LOG_INFO << DLT_MSIN_MTIN_SHIFT);

    // Process message when counter is already at UINT32_MAX
    dlt_daemon_process_message_log_statistic(&daemon, &daemon_local, 0);

    // Verify counter stays at UINT32_MAX (saturation - does NOT overflow to 0)
    EXPECT_EQ(UINT32_MAX, context->log_level_statistics[DLT_LOG_INFO].stat_cnt);

    //Verify size DOES NOT EXCEED UINT32_MAX (saturation behavior)
    EXPECT_LE(context->log_level_statistics[DLT_LOG_INFO].stat_size, UINT32_MAX);

    //Verify size increased but stayed within bounds
    EXPECT_GE(context->log_level_statistics[DLT_LOG_INFO].stat_size, initial_size);

    // Process another message to verify persistent saturation
    dlt_daemon_process_message_log_statistic(&daemon, &daemon_local, 0);

    //Verify counter STILL at UINT32_MAX (no wraparound)
    EXPECT_EQ(UINT32_MAX, context->log_level_statistics[DLT_LOG_INFO].stat_cnt);

    //Verify size still within bounds
    EXPECT_LE(context->log_level_statistics[DLT_LOG_INFO].stat_size, UINT32_MAX);

    // Cleanup
    free(daemon_local.msg.standardheader);
    free(daemon_local.msg.extendedheader);
    dlt_message_free(&daemon_local.msg, 0);
    EXPECT_EQ(0, dlt_daemon_free(&daemon, 0));
}

/* Helper function to create socket pair for testing */
static int create_test_socket_pair(int *client_sock, int *server_sock)
{
    int socks[2];
    if (socketpair(AF_UNIX, SOCK_STREAM, 0, socks) == 0) {
        *client_sock = socks[0];
        *server_sock = socks[1];
        return 0;
    }
    return -1;
}

/* Test send with NULL client pointer - should return error */
TEST(t_dlt_client_send_set_statistics_options, null_client)
{
    DltServiceSetStatisticsOptions options;
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Should handle NULL client gracefully
    EXPECT_EQ(DLT_RETURN_ERROR,
              dlt_client_send_set_statistics_options(NULL, &options));
}

/* Test send with NULL options pointer - should return error */
TEST(t_dlt_client_send_set_statistics_options, null_options)
{
    DltClient client;
    memset(&client, 0, sizeof(DltClient));

    // Should handle NULL options gracefully
    EXPECT_EQ(DLT_RETURN_ERROR,
              dlt_client_send_set_statistics_options(&client, NULL));
}

/* Test send when client socket is not connected - should return error */
TEST(t_dlt_client_send_set_statistics_options, client_not_connected)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    // Initialize client but don't connect
    client.sock = -1; // Invalid socket

    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Should fail due to invalid socket
    EXPECT_EQ(DLT_RETURN_ERROR,
              dlt_client_send_set_statistics_options(&client, &options));
}

/* Test send with invalid service ID - behavior depends on implementation */
TEST(t_dlt_client_send_set_statistics_options, invalid_service_id)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    // Create socket pair for testing
    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Set invalid service ID
    options.service_id = 0xFFFFFFFF; // Invalid
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Should reject invalid service ID
    DltReturnValue ret = dlt_client_send_set_statistics_options(&client, &options);

    // Cleanup
    close(client_sock);
    close(server_sock);

    // May return ERROR or continue (depends on implementation)
    EXPECT_TRUE(ret == DLT_RETURN_ERROR || ret == DLT_RETURN_OK);
}

/* Test valid request to enable statistics collection */
TEST(t_dlt_client_send_set_statistics_options, valid_enable_request)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Prepare valid enable request
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Send request
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Verify message was sent by reading from server socket
    uint8_t buffer[512];
    ssize_t bytes_read = read(server_sock, buffer, sizeof(buffer));
    EXPECT_GT(bytes_read, 0); // Should have received data

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test valid request to disable statistics collection */
TEST(t_dlt_client_send_set_statistics_options, valid_disable_request)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Prepare valid disable request
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_DISABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Send request
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Verify message was sent
    uint8_t buffer[512];
    ssize_t bytes_read = read(server_sock, buffer, sizeof(buffer));
    EXPECT_GT(bytes_read, 0);

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test valid request to reset statistics counters */
TEST(t_dlt_client_send_set_statistics_options, valid_reset_request)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Prepare reset request
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_ENABLED; // Reset
    options.stat_print = STAT_DISABLED;

    // Send request
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Verify message was sent
    uint8_t buffer[512];
    ssize_t bytes_read = read(server_sock, buffer, sizeof(buffer));
    EXPECT_GT(bytes_read, 0);

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test valid print request with output filename specified */
TEST(t_dlt_client_send_set_statistics_options, valid_print_with_filename)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;
    const char *filename = "/tmp/client_test_stats.csv";

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Prepare print request with filename
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_ENABLED; // Print
    strncpy(options.output_file_name, filename, sizeof(options.output_file_name) - 1);
    options.output_file_name[sizeof(options.output_file_name) - 1] = '\0';

    // Send request
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Verify message was sent
    uint8_t buffer[512];
    ssize_t bytes_read = read(server_sock, buffer, sizeof(buffer));
    EXPECT_GT(bytes_read, 0);

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test combined request: enable, reset, and print statistics */
TEST(t_dlt_client_send_set_statistics_options, combined_request)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;
    const char *filename = "/tmp/client_combined.csv";

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Prepare combined request
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;   // Enable
    options.stat_reset = STAT_ENABLED;  // Reset
    options.stat_print = STAT_ENABLED;  // Print
    strncpy(options.output_file_name, filename, sizeof(options.output_file_name) - 1);
    options.output_file_name[sizeof(options.output_file_name) - 1] = '\0';

    // Send request
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Verify message was sent
    uint8_t buffer[512];
    ssize_t bytes_read = read(server_sock, buffer, sizeof(buffer));
    EXPECT_GT(bytes_read, 0);

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test with invalid stat_mode value (not 0 or 1) */
TEST(t_dlt_client_send_set_statistics_options, invalid_stat_mode)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Set invalid stat_mode
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = (DltStatisticsState)255; // Invalid (should be 0 or 1)
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_DISABLED;

    // Function may still send message (validation happens server-side)
    DltReturnValue ret = dlt_client_send_set_statistics_options(&client, &options);

    // Cleanup
    close(client_sock);
    close(server_sock);

    // Client may send anyway, server will validate
    EXPECT_TRUE(ret == DLT_RETURN_OK || ret == DLT_RETURN_ERROR);
}

/* Test with invalid stat_reset value (not 0 or 1) */
TEST(t_dlt_client_send_set_statistics_options, invalid_stat_reset)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Set invalid stat_reset
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = (DltStatisticsState)200; // Invalid
    options.stat_print = STAT_DISABLED;

    // Function may still send message
    DltReturnValue ret = dlt_client_send_set_statistics_options(&client, &options);

    // Cleanup
    close(client_sock);
    close(server_sock);

    EXPECT_TRUE(ret == DLT_RETURN_OK || ret == DLT_RETURN_ERROR);
}

/* Test with invalid stat_print value (not 0 or 1) */
TEST(t_dlt_client_send_set_statistics_options, invalid_stat_print)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Set invalid stat_print
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = (DltStatisticsState)150; // Invalid

    // Function may still send message
    DltReturnValue ret = dlt_client_send_set_statistics_options(&client, &options);

    // Cleanup
    close(client_sock);
    close(server_sock);

    EXPECT_TRUE(ret == DLT_RETURN_OK || ret == DLT_RETURN_ERROR);
}

/* Test print request with filename exceeding maximum length */
TEST(t_dlt_client_send_set_statistics_options, filename_too_long)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Create filename that's too long
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_ENABLED;

    // Fill with maximum characters (no null terminator)
    memset(options.output_file_name, 'x', sizeof(options.output_file_name));

    // Function should handle gracefully (may truncate or reject)
    DltReturnValue ret = dlt_client_send_set_statistics_options(&client, &options);

    // Cleanup
    close(client_sock);
    close(server_sock);

    // Implementation-dependent behavior
    EXPECT_TRUE(ret == DLT_RETURN_OK || ret == DLT_RETURN_ERROR);
}

/* Test print enabled with empty filename - should use default or handle gracefully */
TEST(t_dlt_client_send_set_statistics_options, empty_filename_with_print)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    // Print enabled but empty filename
    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;
    options.stat_mode = STAT_ENABLED;
    options.stat_reset = STAT_DISABLED;
    options.stat_print = STAT_ENABLED; // Print enabled
    options.output_file_name[0] = '\0'; // Empty filename

    // Should still send (server may use default filename)
    EXPECT_EQ(DLT_RETURN_OK,
              dlt_client_send_set_statistics_options(&client, &options));

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test sending multiple consecutive requests - should handle all successfully */
TEST(t_dlt_client_send_set_statistics_options, multiple_consecutive_requests)
{
    DltClient client;
    DltServiceSetStatisticsOptions options;
    int client_sock = 0, server_sock = 0;

    memset(&client, 0, sizeof(DltClient));
    memset(&options, 0, sizeof(DltServiceSetStatisticsOptions));

    ASSERT_EQ(0, create_test_socket_pair(&client_sock, &server_sock));
    client.sock = client_sock;

    options.service_id = DLT_SERVICE_ID_SET_STATISTICS_OPTIONS;

    // Send multiple requests consecutively
    for (int i = 0; i < 5; i++) {
        options.stat_mode = (i % 2 == 0) ? STAT_ENABLED : STAT_DISABLED;
        options.stat_reset = STAT_DISABLED;
        options.stat_print = STAT_DISABLED;

        EXPECT_EQ(DLT_RETURN_OK,
                  dlt_client_send_set_statistics_options(&client, &options));

        // Read and discard response
        uint8_t buffer[512];
        read(server_sock, buffer, sizeof(buffer));
    }

    // Cleanup
    close(client_sock);
    close(server_sock);
}

/* Test get_next_tab_from_string with NULL pointer - should return 0 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, null_pointer)
{
    int result = dlt_daemon_statistic_get_next_tab_from_string(NULL);
    EXPECT_EQ(0, result);
}

/* Test get_next_tab_from_string with empty string - should return 1 (0/tablength + 1) */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, empty_string)
{
    char empty[] = "";
    int result = dlt_daemon_statistic_get_next_tab_from_string(empty);
    EXPECT_EQ(1, result);
}

/* Test get_next_tab_from_string with short string - should return 1 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, short_string)
{
    char short_str[] = "A";
    int result = dlt_daemon_statistic_get_next_tab_from_string(short_str);
    EXPECT_EQ(1, result);
}

/* Test get_next_tab_from_string with string equal to tablength - should return 2 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, string_equal_to_tablength)
{
    /* tablength is 15 (TABLEN_1), so string of 15 chars should give (15/15)+1 = 2 */
    char str_15[] = "123456789012345";  /* 15 characters */
    EXPECT_EQ(15, strlen(str_15));
    int result = dlt_daemon_statistic_get_next_tab_from_string(str_15);
    EXPECT_EQ(2, result);
}

/* Test get_next_tab_from_string with string of tablength-1 - should return 1 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, string_less_than_tablength)
{
    /* tablength is 15, so string of 14 chars should give (14/15)+1 = 1 */
    char str_14[] = "12345678901234";  /* 14 characters */
    EXPECT_EQ(14, strlen(str_14));
    int result = dlt_daemon_statistic_get_next_tab_from_string(str_14);
    EXPECT_EQ(1, result);
}

/* Test get_next_tab_from_string with string of 2*tablength - should return 3 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, string_two_tablengths)
{
    /* tablength is 15, so string of 30 chars should give (30/15)+1 = 3 */
    char str_30[] = "123456789012345123456789012345";  /* 30 characters */
    EXPECT_EQ(30, strlen(str_30));
    int result = dlt_daemon_statistic_get_next_tab_from_string(str_30);
    EXPECT_EQ(3, result);
}

/* Test get_next_tab_from_string with string of 2*tablength-1 - should return 2 */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, string_two_tablengths_minus_one)
{
    /* tablength is 15, so string of 29 chars should give (29/15)+1 = 2 */
    char str_29[] = "12345678901234512345678901234";  /* 29 characters */
    EXPECT_EQ(29, strlen(str_29));
    int result = dlt_daemon_statistic_get_next_tab_from_string(str_29);
    EXPECT_EQ(2, result);
}

/* Test get_next_tab_from_string with very long string - should handle correctly */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, very_long_string)
{
    /* Create a string of 500 characters */
    char long_str[501];
    memset(long_str, 'a', 500);
    long_str[500] = '\0';
    EXPECT_EQ(500, strlen(long_str));

    /* tablength is 15, so 500/15 = 33, +1 = 34 */
    int result = dlt_daemon_statistic_get_next_tab_from_string(long_str);
    EXPECT_EQ(34, result);
}

/* Test get_next_tab_from_string with string containing spaces */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, string_with_spaces)
{
    char str_spaces[] = "Hello World     ";  /* 16 characters */
    EXPECT_EQ(16, strlen(str_spaces));

    /* 16/15 = 1, +1 = 2 */
    int result = dlt_daemon_statistic_get_next_tab_from_string(str_spaces);
    EXPECT_EQ(2, result);
}

/* Test get_next_tab_from_string with multiple consecutive calls - ensure no state pollution */
TEST(t_dlt_daemon_statistic_get_next_tab_from_string, consecutive_calls)
{
    char str1[] = "A";
    char str2[] = "123456789012345";  /* 15 chars */
    char str3[] = "123456789012345123456789012345";  /* 30 chars */

    int result1 = dlt_daemon_statistic_get_next_tab_from_string(str1);
    int result2 = dlt_daemon_statistic_get_next_tab_from_string(str2);
    int result3 = dlt_daemon_statistic_get_next_tab_from_string(str3);

    /* Verify consistent results */
    EXPECT_EQ(1, result1);
    EXPECT_EQ(2, result2);
    EXPECT_EQ(3, result3);

    /* Call again to verify no state changes */
    int result1_again = dlt_daemon_statistic_get_next_tab_from_string(str1);
    EXPECT_EQ(1, result1_again);
}

/* Test add_spaces_to_string with NULL text pointer - should return error */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, null_text_pointer)
{
    char **text = NULL;
    int result = dlt_daemon_statistic_add_spaces_to_string(text, 5, 256);
    EXPECT_EQ(-1, result);
}

/* Test add_spaces_to_string with invalid buffer size (0) - should return error */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, invalid_buffer_size_zero)
{
    char buffer[256] = "Hello";
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 5, 0);
    EXPECT_EQ(-1, result);
}

/* Test add_spaces_to_string with buffer size exceeding max - should return error */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, buffer_size_exceeds_max)
{
    char buffer[512];
    memset(buffer, 'a', 512);
    char *text = buffer;
    /* DLT_LOG_STATISTICS_MAX_PRINT_LENGTH is 256 */
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 5, 257);
    EXPECT_EQ(-1, result);
}

/* Test add_spaces_to_string with num <= 0 - should return success without adding spaces */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, num_zero)
{
    char buffer[256] = "Hello";
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 0, 256);
    EXPECT_EQ(0, result);
    EXPECT_STREQ(text, "Hello");
}

/* Test add_spaces_to_string with negative num - should return success without adding spaces */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, num_negative)
{
    char buffer[256] = "Test";
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, -1, 256);
    EXPECT_EQ(0, result);
    EXPECT_STREQ(text, "Test");
}

/* Test add_spaces_to_string with single space on empty buffer - should succeed */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, add_one_space_to_empty)
{
    char buffer[256];
    memset(buffer, 0, sizeof(buffer));
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 1, 256);
    EXPECT_EQ(0, result);
    EXPECT_EQ(1, strlen(buffer));
    EXPECT_EQ(' ', buffer[0]);
}

/* Test add_spaces_to_string with single space to existing text - should succeed */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, add_one_space_to_text)
{
    char buffer[256] = "Hello";
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 1, 256);
    EXPECT_EQ(0, result);
    EXPECT_STREQ(buffer, "Hello ");
}

/* Test add_spaces_to_string with multiple spaces - should succeed */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, add_multiple_spaces)
{
    char buffer[256] = "Test";
    char *text = buffer;
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 5, 256);
    EXPECT_EQ(0, result);
    EXPECT_STREQ(buffer, "Test     ");
    EXPECT_EQ(9, strlen(buffer));
}

/* Test add_spaces_to_string buffer overflow - should fail and preserve buffer */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, buffer_overflow)
{
    char buffer[10] = "Hello";
    char *text = buffer;
    /* Trying to add 10 spaces to a 10-byte buffer with "Hello" (5 chars) will overflow */
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 10, 10);
    EXPECT_EQ(-1, result);
}

/* Test add_spaces_to_string at exact buffer boundary - should succeed */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, exact_buffer_fit)
{
    char buffer[11];
    memset(buffer, 0, sizeof(buffer));
    strcpy(buffer, "Hello");
    char *text = buffer;
    /* Buffer has 11 bytes: "Hello" (5) + 5 spaces + null terminator = 11 bytes exactly */
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 5, 11);
    EXPECT_EQ(0, result);
    EXPECT_STREQ(buffer, "Hello     ");
}

/* Test add_spaces_to_string with long existing string - should handle correctly */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, long_existing_string)
{
    char buffer[256];
    memset(buffer, 'x', 200);
    buffer[200] = '\0';
    char *text = buffer;

    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 20, 256);
    EXPECT_EQ(0, result);
    EXPECT_EQ(220, strlen(buffer));
}

/* Test add_spaces_to_string consecutive calls - should accumulate spaces */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, consecutive_calls)
{
    char buffer[256] = "Start";
    char *text = buffer;

    int result1 = dlt_daemon_statistic_add_spaces_to_string(&text, 2, 256);
    EXPECT_EQ(0, result1);
    EXPECT_STREQ(buffer, "Start  ");

    int result2 = dlt_daemon_statistic_add_spaces_to_string(&text, 3, 256);
    EXPECT_EQ(0, result2);
    EXPECT_STREQ(buffer, "Start     ");
    EXPECT_EQ(10, strlen(buffer));
}

/* Test add_spaces_to_string with full buffer - should fail when trying to add more */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, full_buffer_overflow_attempt)
{
    char buffer[21];
    memset(buffer, 0, sizeof(buffer));
    strcpy(buffer, "A123456789B123456789");  /* 20 chars */
    char *text = buffer;

    /* Try to add 1 more space to a full 21-byte buffer (only room for null terminator) */
    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 1, 21);
    EXPECT_EQ(-1, result);
}

/* Test add_spaces_to_string with newline in buffer - should handle correctly */
TEST(t_dlt_daemon_statistic_add_spaces_to_string, string_with_content)
{
    char buffer[256] = "Info";
    char *text = buffer;

    int result = dlt_daemon_statistic_add_spaces_to_string(&text, 15, 256);
    EXPECT_EQ(0, result);
    EXPECT_EQ(19, strlen(buffer));
}

int main(int argc, char **argv)
{
    ::testing::InitGoogleTest(&argc, argv);
    ::testing::FLAGS_gtest_break_on_failure = false;
    return RUN_ALL_TESTS();
}
#endif
